/* 
 * File:   DevShell.cpp
 * Author: david
 * 
 * Created on November 26, 2012, 2:16 PM
 */

#include "DevShell.h"
#include <limits>
using namespace std;

DevShell::DevShell()
{
}

DevShell::DevShell(const DevShell& orig)
{
}

DevShell::~DevShell()
{
}

void DevShell::init(OverlapsGraph *g, Pairing *p, ReadsStorage *r, ReadsLayout *l) {
    G = g;
    P = p;
    R = r;
    L = l;
    checkDistances = true;
    minNPair = 5;
    minRatio = 0.95;
    if (!BKW.isAllocated())
        BKW.init(G);
}

void DevShell::prompt()
{
    string str,buffer;
  
    bool finished = false;

    do
    {
        cout << "edenaDev> ";
        args.clear();
        getline(cin, buffer);
        iss.clear();
        iss.str(buffer);
        do
        {
            iss >> str;
            if (!iss.fail())
                args.push_back(str);
        }
        while (!iss.fail());

        if (args.size() == 0)
            continue;

        if (args[0] == "g")
        {
            drawGraph();
        }
        else if (args[0] == "t")
        {
            growSearchTree();
        }
        else if (args[0] == "e")
        {
            elongateSearchTree();
        }
       
        else if (args[0] == "t2s")
        {
            Tree2seq();
        }
        else if (args[0] == "p2s")
        {
            path2Seq();
        }
        else if (args[0] == "rp")
        {
            reversePath();
        }
        else if (args[0] == "l")
        {
            printReadsLayout();
        }
        else if (args[0] == "h" || args[0] == "H" || args[0] == "help")
        {
            printHelp(cerr);
        }
        else if (args[0] == "findRead")
        {
            locateRead();
        }
        else if (args[0]=="checkD")
        {
            setCheckDistances();
        }
        else if (args[0] == "q" || args[0] == "Q" || args[0] == "quit")
        {
            cerr << "bye" << endl;
            finished=true;
        }
        else
        {
            whatDoYouMean(cerr);
        }

    }
    while (finished == false);
}

void DevShell::printHelp(ostream &out)
{
     cout << "DEV SHELL COMMANDS:\n"
    << "   g <ID> <depth>       draw graph around node <ID>\n"
    << "                        (require graphviz). 'depth' is optional.\n"
    << "                        The graph is written to the file \"OSG.ps\".\n"
    << "   t <PATH>             Initialize the search Tree with a path.\n"
    << "                        PATH can be prefixed with \'R\' to specify\n"
    << "                        the path reverse-complemented. The tree is\n"
    << "                        written to the file \"searchTree.ps\".\n"
    << "   t <no argument>      Grow the search tree by one more stage.\n"
    << "   e <treenode_ID>      Elongate the determined path using treenode_ID.\n"
    << "                        treenode_ID must be one of the candidate.\n"
    << "                        This command may updates the file \"searchTree.ps\".\n"
    << "   e <no argument>      Same as above, but using the candidate determined\n"
    << "                        by the extension algorithm (if even).\n"
    << "   t2s                  Extract the sequence corresponding to the backward\n"
    << "                        walker determined path.\n"
    << "   checkD yes/no        Set the 'check PE distance' flag:\n"
    << "                        yes: pairs must be connected within allowed range\n"
    << "                             to be matched.\n"
    << "                        no: pairs are matched as long as a path between the\n"
    << "                        the two reads exists.\n"         
    << "   rp <PATH>            Display the PATH reverse-complemented.\n"
    << "   p2s <PATH>           Display the sequence corresponding\n"
    << "                        to a path in the graph. The path is required to\n"
    << "                        be valid and unambiguous to be displayed.\n"
    << "   l <ID>               Display the reads layout in node <ID>.\n"
    << "                        !!can be long to complete for long nodes...!!\n"         
    << "   l <ID> <maxD>        Display the layout up to a distance of maxD.\n"         
    << "   l <ID> <begin> <end> Display the layout for coordinates begin..end.\n"
    << "   findRead <readID>    Display the Node ID and position of readID.\n"
    << "   q                    bye!\n\n"
    << "   <ID> can be specified as:\n"
    << "               1234        (node 1234)\n"
    << "              !1234        (same but reverse comp)\n"
    << "   <readID> is the read number, as displayed by the 'l' (layout) command\n\n"
    << "   <PATH> is a '-' separated list of nodeIDs as displayed in the '.lay'\n"
    << "                   file. A single node <ID> is also considered as a path\n";
}

void DevShell::growSearchTree()
{
    
  
    if (args.size()>1)
    {
        G->nodeListToEPath(args.at(1), ePath);
        BKW.initWalker(ePath, minNPair, minRatio, checkDistances, 10000);
        BKW.stepElongate(0, chosen);
        BKW.dot("searchTree");
        if (chosen!=0)
            cerr << "Chosen candidate: " << chosen << endl;
        else
            cerr << "no determined elongation" << endl;
        
    }

    else if (args.size() == 1) {
        if (BKW.isInitialized() == false) {
            cerr << "you must first initialize the backward walker with a path" << endl;
            return;
        }
        BKW.stepElongate(0, chosen);
        BKW.dot("searchTree");

        if (chosen != 0)
            cerr << "Chosen candidate: " << chosen << endl;
        else
            cerr << "no determined elongation" << endl;

    }

};

void DevShell::elongateSearchTree() {
    
    unsigned int candidate;
    if (BKW.isInitialized() == false) {
        cerr << "You must first initialize the search tree with a path" << endl;
        return;
    }

    if (args.size() == 1) {
        if (chosen != 0) {
            cerr << "Chosen candidate: " << chosen << endl;
            BKW.acceptChoice(chosen);
            chosen=0;
                BKW.dot("searchTree");
        }
        else
        {//try another tree stage
               BKW.stepElongate(0, chosen);
               if (chosen != 0)
               {
                    cerr << "Chosen candidate: " << chosen << endl;
                    BKW.acceptChoice(chosen);
                    chosen=0;
               }
               else
               {
                   cerr << "no chosen candidate" << endl;
               }
                   
               BKW.dot("searchTree");
        }
    } else if (args.size() > 1) {
        iss.clear();
        iss.str(args.at(1));
        iss >> candidate;

        if (BKW.acceptChoice(candidate) == 0) {
            cout << "tree_node: " << candidate << " is not a candidate" << endl;
        } else {
            cerr << "Chosen candidate: " << chosen << endl;
            BKW.dot("searchTree");
        }
    }
};

void DevShell::Tree2seq() {
    
    vector<unsigned int> epath;
    vector<unsigned int> path;
    vector<bool> dir;
    string s;
    ostringstream oss;
    ofstream out;

    if (BKW.isInitialized() == false) {
        cerr << "You must first initialize the search tree with a path" << endl;
        return;
    }

    BKW.getPath(epath);
    G->ePathToNodePath(epath, pathNode, pathDir);
    s = G->ePathToSeq(epath);
    
    for (size_t i = 1; i < pathNode.size(); i++) {
        if (!pathDir[i])
            oss << '!';
        oss << pathNode[i];
        if (i < pathNode.size() - 1)
            oss << '-';
    }

    if (s != "") {
        out.open("out.dev");
        cerr << "SearchTreeToSeq (l=" << pathNode[0] << ") " << oss.str();
        out << ">SearchTreeToSeq (l=" << pathNode[0] << ") " << oss.str();
        cerr << '\n';
        out << '\n';
        lineWrap(cerr, s, 70);
        lineWrap(out, s, 70);
        out.close();
    }

}

void DevShell::drawGraph()
{
    parseNodeId(args.at(1));
    unsigned int gDepth = 8;
    if (nodeID == 0)
    {
        whatDoYouMean(cerr);
        return;
    }
    
    if (args.size() == 3)
    {
        iss.clear();
        iss.str(args.at(2));
        iss >> gDepth;
    }

    G->nodesTab[nodeID].dotLocalGraph(gDepth, "OSG");
}

void DevShell::path2Seq()
{
    
    if (args.size() < 2)
        return;
    
    G->nodeListToEPath(args.at(1), ePath);
    string str = G->ePathToSeq(ePath);
    ofstream out;
    if (str != "") {
        out.open("out.dev");
        cerr << "pathToSeq (l=" << str.size() << ") " << args.at(1);
        out << ">pathToSeq (l=" << str.size() << ") " << args.at(1);
        cerr << '\n';
        out << '\n';
        lineWrap(cerr, str, 70);
        lineWrap(out, str, 70);
        out.close();
    }
    else
        cout << "invalid path" << endl;
    
}

void DevShell::reversePath() {
    
    string pp = "R";
    vector<unsigned int> ePath;
    vector<unsigned int> nn;
    vector<bool> dd;
    pp += args.at(1);
    G->nodeListToEPath(pp, ePath);
    G->ePathToNodePath(ePath, nn, dd);
    for (unsigned int i = 1; i < nn.size(); i++) {
        if (!dd[i])
            cout << "!";
        cout << nn[i];
        if (i < nn.size() - 1)
            cout << "-";
    }
    cout << endl;

};

void DevShell::printReadsLayout()
{

    parseNodeId(args.at(1));

    if (nodeID == 0) {
        whatDoYouMean(cerr);
       return;
    }
    unsigned int maxD = numeric_limits<unsigned int>::max();
    unsigned int start = 0;
    
    if (args.size() == 3) {
        iss.clear();
        iss.str(args.at(2));
        iss >> maxD;
        start=0;
    }
    else if (args.size() == 4) {
        iss.clear();
        iss.str(args.at(2));
        iss >> start;
        iss.clear();
        iss.str(args.at(3));
        iss >> maxD;
    }

    L->print(G->nodesTab[nodeID].getLayout(), cerr, nodeDir, start, maxD, P);
    cout << endl;
};

void DevShell::locateRead() {


    unsigned int readId=0;
    bool dir=true;
    
    if (args.size()<2)
    {
        whatDoYouMean(cerr);
        return;
    }
    iss.clear();
    iss.str(args.at(1));
    iss >> readId; //read ID
    if (iss.fail()) {
        whatDoYouMean(cerr);
        return;
    }

    if (readId < 1 || readId >= R->getEffectiveNReads() + 1) {
        cout << "<readID> out of range (1.." << R->getEffectiveNReads() << ")\n";
        return;
    }
    if (L->getDirection(readId) == false)
        dir = false;
    cerr << L->getNodeId(readId);
    if (!dir)
        cerr << " < ";
    else
        cerr << " > ";
    cerr << "pos: " << L->getPosition(readId) << endl;
}

void DevShell::setCheckDistances()
{
    if (args.size() != 2)
    {
        cerr << "possible paramters are 'yes' or 'no'" << endl;
        return;
    }
    
    if (args[1]=="yes")
    {
        checkDistances=true;
        cerr << "check PE distance flag: YES" << endl;
    }
    else if (args[1]=="no")
    {
        checkDistances=false;
        cerr << "check PE distance flag: NO" << endl;
    }
    else
        cerr << "possible paramters are 'yes' or 'no'" << endl;
    
    BKW.setCheckDistances(checkDistances);
}

void DevShell::parseNodeId(string buffer)
{
    if (buffer[0] == '!')//reverse
    {
        nodeDir = false;
        buffer = buffer.substr(1);
    }
    else
        nodeDir = true;

    iss.clear();
    iss.str(buffer);
    iss >> nodeID;
    if (iss.fail())
        nodeID = 0;

    if (nodeID < 1 || nodeID > G->getNNodes())
    {
        cout << "<nodeID> out of range (1.." << G->getNNodes() << ")\n";
        nodeID = 0;
    }
}